package com.miketheshadow.autoregister;

import com.miketheshadow.autoregister.annotations.CommandLoader;
import org.apache.commons.lang.NotImplementedException;
import org.bukkit.Bukkit;
import org.bukkit.command.CommandExecutor;
import org.bukkit.command.PluginCommand;
import org.bukkit.event.Listener;
import org.bukkit.plugin.Plugin;
import org.bukkit.plugin.PluginManager;

import java.io.File;
import java.io.FileInputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.Set;
import java.util.jar.JarInputStream;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;

/**
 * The core class of this tool. Automatically reads the jar file and
 * attempts to register any listener and command available.
 * Commands need to be annotated with {@link CommandLoader}
 * Listeners need to extend Spigots Listener class.
 */
public final class AutoRegister {

    Plugin plugin;
    String packageName;
    Set<Class<?>> classes;
    boolean force = false;
    boolean debugLogging = false;

    /**
     *
     * @param plugin Your plugin. Do not pass in another plugin but your own into this method, or
     *               you will end up with nothing working
     * @param packageName A package path using . as the separator. Example: com.miketheshadow.autoregister
     *                    Where autoregister is the base package directory.
     *                    Do not pass in something like com. or com.name as this could have the unintended
     *                    effect of registering any library's listeners as well.
     */
    public AutoRegister(Plugin plugin, String packageName) {
        this.plugin = plugin;
        this.packageName = packageName;
        this.classes = collectAllClasses();
    }

    /**
     * Enable debug messages.
     * Currently, the names of everything registered and the sizes/amounts of classes loaded are all that
     * is logged.
     * @return The instance of {@link AutoRegister}.
     */
    public AutoRegister enableDebugMessages() {
        this.debugLogging = true;
        return this;
    }

    /**
     * Enables the force loading of all classes.
     * In some circumstances it's possible for things to not work as intended.
     * force loading all classes will create a more significant delay on the
     * speed of loading and should only be used if listeners are not registering properly.
     * @return The instance of {@link AutoRegister}.
     */
    public AutoRegister forceLoadAllClasses() {
        this.force = true;
        return this;
    }

    /**
     * This is what should be used by default.
     * Unless you have your own custom auto-registry this
     * method will handle the registering of commands and listeners for you.
     */
    public void defaultSetup() {
        try {
            debugLog("Registering listeners...");
            registerListeners();
            debugLog("Registering commands....");
            registerCommands();
            debugLog("Setup complete!");
        } catch (Exception e) {
            Bukkit.getLogger().severe("Unable to register events with message: " + e.getMessage());
            e.printStackTrace();
        }

    }

    /**
     * Registers all listener classes
     * @throws Exception This should probably never throw, but if it does contact me.
     */
    public void registerListeners() throws Exception {
        Set<Class<?>> clazzes = getListeners();
        PluginManager manager = Bukkit.getServer().getPluginManager();
        for(Class<?> clazz : clazzes) {
            debugLog("Registering listener: " + clazz.getName());
            Listener listener = (Listener) clazz.getDeclaredConstructor().newInstance();
            manager.registerEvents(listener,plugin);

            try {
                Field field = listener.getClass().getField("plugin");
                field.setAccessible(true);
                field.set(Plugin.class,plugin);
            } catch (NoSuchFieldException | IllegalAccessException ignored) {}

        }
    }

    /**
     *
     * @throws Exception Like above this will probably never throw (if you're annotating the right things),
     * but if it does send me a screenshot because that would be pretty breaking.
     */
    public void registerCommands() throws Exception {
        Set<Class<?>> annotated = getClassesAnnotatedWith(CommandLoader.class);

        for(Class<?> clazz : annotated) {
            CommandLoader commandAnnotation = clazz.getAnnotation(CommandLoader.class);
            if(commandAnnotation == null) continue;
            String commandName = commandAnnotation.commandName();
            PluginCommand command = Bukkit.getServer().getPluginCommand(commandName);
            debugLog("Registering command: " + commandName + " from class " + clazz.getName());
            if(command == null) {
                throw new NotImplementedException("Missing plugin.yml registration for command: " + commandName);
            }
            command.setExecutor((CommandExecutor) clazz.getDeclaredConstructor().newInstance());
        }
    }

    /**
     * Gets any class in the package annotated with annotation
     * @param <A> Specifies the class is an annotation.
     * @param annotation The annotation to filter the classes by.
     * @return All classes annotated with the parameter.
     */
    public <A extends Annotation> Set<Class<?>> getClassesAnnotatedWith(Class<A> annotation) {
        if(!annotation.isAnnotation()) {
            throw new IllegalStateException("Class " + annotation.getName() + " is not an annotation!");
        }
        Set<Class<?>> annotated = classes.stream().filter(clazz -> clazz.getAnnotation(annotation) != null).collect(Collectors.toSet());
        debugLog("Found: " + annotated.size() + " classes annotated with " + annotation.getName());
        return annotated;
    }

    /**
     * @return returns a Set of classes that implement Spigot's Listener class
     */
    public Set<Class<?>> getListeners() {
        Set<Class<?>> listeners = classes.stream().filter(Listener.class::isAssignableFrom).collect(Collectors.toSet());
        debugLog("Found: " + listeners.size());
        return listeners;
    }

    /**
     * @return a set of every class in your project.
     */
    public Set<Class<?>> getClasses() {
        return classes;
    }


    private boolean initialPackageFound = false;

    /**
     * This opens the jar file to basically read the contents.
     * I can't imagine the jar is ever unreadable, but you never know these days.
     * To get all the current packages use {@link AutoRegister#getClasses}
     * @return All the classes within the package.
     */
    private Set<Class<?>> collectAllClasses() {
        String searchName = packageName.replaceAll("[.]", "/");
        ClassLoader classLoader = plugin.getClass().getClassLoader();
        Set<Class<?>> classes = new HashSet<>();
        try {
            File currentFile = new File(plugin.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
            JarInputStream jarInputStream = new JarInputStream(new FileInputStream(currentFile));
            while (true) {
                ZipEntry entry = jarInputStream.getNextEntry();
                if (entry == null) break;
                String name = entry.getName();
                if(!force) {
                    String compare;
                    if(name.length() >= searchName.length()) {
                        compare = name.substring(0,searchName.length() - 1);
                        initialPackageFound = true;
                    } else {
                        compare = name;
                    }
                    if(!searchName.contains(compare) && !initialPackageFound) break;
                }

                if (name.contains(searchName) && name.endsWith(".class")) {
                    classes.add(
                            classLoader.loadClass(name.replace(".class", "")
                                    .replaceAll("/", ".")));
                }
            }
            return classes;
        } catch (Exception e) {
            e.printStackTrace();
        }
        debugLog("Total classes loaded: " + classes.size());
        return classes;
    }

    private void debugLog(String message) {
        if(debugLogging) {
            Bukkit.getLogger().info(message);
        }
    }

}
