package com.miketheshadow.cryptolib.request.bulkrequest;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.miketheshadow.cryptolib.CryptoLib;
import com.miketheshadow.cryptolib.request.IRequestError;
import com.miketheshadow.cryptolib.request.RequestException;
import okhttp3.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

public class BulkRequest<T> {
    Logger logger = LoggerFactory.getLogger(BulkRequest.class);
    private final List<? extends Requestable> requestables;

    private final Class<?> mapping;
    private final String nodeString;
    private final boolean retryOnError;
    private int delay = 0;
    private int errorDelay = 0;
    private boolean debug = false;
    private int increaseBy = 100;

    public BulkRequest(CryptoLib lib, List<? extends Requestable> requestables, Class<?> mapping, boolean retryOnError) {
        this.nodeString = lib.getNodeString();
        this.requestables = requestables;
        this.mapping = mapping;
        this.retryOnError = retryOnError;
    }

    public BulkRequest(String nodeString, List<? extends Requestable> requestables, Class<?> mapping, boolean retryOnError) {
        this.nodeString = nodeString;
        this.requestables = requestables;
        this.mapping = mapping;
        this.retryOnError = retryOnError;
    }

    public void withDelay(int seconds) {
        this.delay = seconds;
    }

    public void withExtraDebug() {
        this.debug = true;
    }

    public void setDelayOnError(int delay) {
        this.errorDelay = delay;
    }

    public void setIncreaseBy(int amount) {
        this.increaseBy = amount;
    }

    public List<T> get() throws RequestException {

        List<T> result = new ArrayList<>();
        double totalCompleted = 0;
        for (String requestString : generateBulkRequests()) {
            while (true) {
                try {
                    MediaType mediaType = MediaType.parse("application/json");
                    RequestBody body = RequestBody.create(mediaType, requestString);
                    Request request = new Request.Builder()
                            .url(nodeString)
                            .method("POST", body)
                            .addHeader("Content-Type", "application/json")
                            .build();
                    List<T> values = executeRequest(request);
                    if (values == null) throw new RequestException("Unable to update Bulk Requests!");
                    if (debug) {
                        totalCompleted += values.size();
                        logger.info("adding " + values.size() + " values " + totalCompleted + "/" + requestables.size() + " " + (totalCompleted / requestables.size() * 100d) + "%");
                    }
                    boolean error = false;
                    for (T req : values) {
                        if (req instanceof IRequestError requestError) {
                            if (requestError.getError() != null && requestError.getError().message != null) {
                                logger.warn("generic request generated an error: " + requestError.getError().message + " retrying...");
                                error = true;
                            }
                        }
                    }
                    if (error) {
                        if (errorDelay > 0) {
                            try {
                                Thread.sleep(errorDelay * 1000L);
                            } catch (InterruptedException e) {
                                throw new RuntimeException(e);
                            }
                        }
                        continue;
                    }
                    result.addAll(values);
                    break;
                } catch (RequestException requestException) {
                    if (!retryOnError) throw new RequestException(requestException);
                    logger.warn("Bulk Request errored however retryOnError is true: " + requestException.getMessage());
                }
                if (delay > 0) {
                    try {
                        Thread.sleep(delay * 1000L);
                    } catch (InterruptedException e) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }
        return result;
    }

    private List<String> generateBulkRequests() {

        List<? extends List<? extends Requestable>> requestLists = Lists.partition(requestables, increaseBy / requestables.get(0).requestSize());

        List<String> requestStrings = new ArrayList<>();

        AtomicInteger atomicInteger = new AtomicInteger(0);
        for (List<? extends Requestable> list : requestLists) {
            requestStrings.add(generateRequest(list, atomicInteger));
        }
        return requestStrings;
    }

    private String generateRequest(List<? extends Requestable> requests, AtomicInteger startIndex) {
        Joiner joiner = Joiner.on(",");
        return "[" + joiner.join(requests.stream().map((requestable) -> requestable.getAsRequest(startIndex.getAndIncrement())).toList()) + "]";
    }

    private List<T> executeRequest(Request request) {
        OkHttpClient client = new OkHttpClient().newBuilder().build();
        try (Response response = client.newCall(request).execute()) {
            String bodyString = response.body().string();
            ObjectMapper mapper = new ObjectMapper();
            JavaType type = getCollectionType(mapper, List.class, mapping);
            return mapper.readValue(bodyString, type);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    public static JavaType getCollectionType(ObjectMapper mapper, Class<?> collectionClass, Class<?>... elementClasses) {
        return mapper.getTypeFactory().constructParametricType(collectionClass, elementClasses);
    }

}