package com.miketheshadow.cryptolib.utils.bulkrequest;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.miketheshadow.cryptolib.CryptoLib;
import com.miketheshadow.cryptolib.containers.GenericRequest;
import com.miketheshadow.cryptolib.containers.IRequestError;
import com.miketheshadow.cryptolib.utils.RequestException;
import okhttp3.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

public class BulkRequest<T> {
    Logger logger = LoggerFactory.getLogger(BulkRequest.class);
    private final List<? extends Requestable> requestables;

    private final Class<?> mapping;
    private final CryptoLib lib;
    private final boolean retryOnError;
    private int delay = 0;
    private boolean debug = false;

    public BulkRequest(CryptoLib lib, List<? extends Requestable> requestables, Class<?> mapping, boolean retryOnError) {
        this.lib = lib;
        this.requestables = requestables;
        this.mapping = mapping;
        this.retryOnError = retryOnError;
    }

    public void withDelay(int seconds) {
        this.delay = seconds;
    }

    public void withExtraDebug() {
        this.debug = true;
    }

    public List<T> get() throws RequestException {

        List<T> result = new ArrayList<>();

        for (String requestString : generateBulkRequests()) {
            while (true) {
                try {
                    MediaType mediaType = MediaType.parse("application/json");
                    RequestBody body = RequestBody.create(mediaType, requestString);
                    Request request = new Request.Builder()
                            .url(lib.getNodeString())
                            .method("POST", body)
                            .addHeader("Content-Type", "application/json")
                            .build();
                    List<T> values = executeRequest(request);
                    if (values == null) throw new RequestException("Unable to update Bulk Requests!");
                    if (debug) {
                        logger.info("adding " + values.size() + " values");
                    }
                    // ADD MORE ERRORING THINGS HERE
                    boolean error = false;
                    for (T req : values) {
                        if (req instanceof IRequestError requestError) {
                            if (requestError.getError() != null && requestError.getError().message != null) {
                                logger.warn("generic request generated an error: " + requestError.getError().message + " retrying...");
                                error = true;
                            }
                        }
                    }
                    if(error) continue;
                    result.addAll(values);
                } catch (RequestException requestException) {
                    if (!retryOnError) throw new RequestException(requestException);
                    logger.warn("Bulk Request errored however retryOnError is true: " + requestException.getMessage());
                }
                if (delay > 0) {
                    try {
                        Thread.sleep(delay * 1000L);
                    } catch (InterruptedException e) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }
        return result;
    }

    // Since nodes limit request sizes to 500 per we need to make sure that cap is never reached
    private List<String> generateBulkRequests() {

        List<? extends List<? extends Requestable>> requestLists = Lists.partition(requestables, 500 / requestables.get(0).requestSize());

        List<String> requestStrings = new ArrayList<>();

        AtomicInteger atomicInteger = new AtomicInteger(0);
        for (List<? extends Requestable> list : requestLists) {
            requestStrings.add(generateRequest(list, atomicInteger));
        }
        return requestStrings;
    }

    private String generateRequest(List<? extends Requestable> requests, AtomicInteger startIndex) {
        Joiner joiner = Joiner.on(",");
        return "[" + joiner.join(requests.stream().map((requestable) -> requestable.getAsRequest(startIndex.getAndIncrement())).toList()) + "]";
    }

    private List<T> executeRequest(Request request) {
        OkHttpClient client = new OkHttpClient().newBuilder().build();
        try (Response response = client.newCall(request).execute()) {
            String bodyString = response.body().string();
            ObjectMapper mapper = new ObjectMapper();
            JavaType type = getCollectionType(mapper, List.class, mapping);
            return mapper.readValue(bodyString, type);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    public static JavaType getCollectionType(ObjectMapper mapper, Class<?> collectionClass, Class<?>... elementClasses) {
        return mapper.getTypeFactory().constructParametricType(collectionClass, elementClasses);
    }

}